/*
 * Copyright (c) 1999-2001 Lutris Technologies, Inc. All Rights
 * Reserved.
 * 
 * This source code file is distributed by Lutris Technologies, Inc. for
 * use only by licensed users of product(s) that include this source
 * file. Use of this source file or the software that uses it is covered
 * by the terms and conditions of the Lutris Enhydra Development License
 * Agreement included with this product.
 * 
 * This Software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
 * ANY KIND, either express or implied. See the License for the specific terms
 * governing rights and limitations under the License.
 * 
 * Contributor(s):
 * 
 * $Id: LookAheadReader.java,v 1.1.1.1 2001/01/05 05:11:18 sese0235 Exp $
 */

package de.kxml.io;

import java.io.*;
import java.util.*;

/**
 * Like a reader, but has a peek function that shows the next
 * char. Unlike reader, it does not throw IOExceptions. Useful for
 * building all kind of parsers
 */
public class LookAheadReader extends Reader {
    protected Reader reader;
    int		     next;
    int		     lineNumber;
    int		     columnNumber;

    /**
     * Creates a new LookAheadReader instance based on the
     * given reader
     */
    public LookAheadReader(Reader reader) throws IOException {
	this.reader = reader;

	read();
    }

    /**
     * closes the LookAheadReader by closing the underlying reader
     */
    public void close() throws IOException {
	try {
	    reader.close();
	} catch (IOException e) {
	    throw new ParseException(e, this);
	} 
    } 

    /**
     * returns true when the end of the stream is reached
     */
    public boolean eof() {
	return next == -1;
    } 

    /**
     * returns the current line number
     */
    public int getLineNumber() {
	return lineNumber;
    } 

    /**
     * returns the current column number
     */
    public int getColumnNumber() {
	return columnNumber;
    } 

    /**
     * skips unicode whitespace chars
     */
    public void skipWhitespace() throws IOException {
	while (next != -1 && next <= 32) {
	    read();
	} 
    } 

    /**
     * unlike Reader.read, this method does not return
     * before all bytes are read.
     */
    public int read(char[] buf, int off, int len) throws IOException {
	int ok = 0;

	while (ready() && ok < len) {
	    int i = read();

	    if (i == -1) {
		break;
	    } 

	    buf[off++] = (char) i;
	    ok++;
	} 

	return ok;
    } 

    /**
     * Method declaration
     *
     *
     * @param count
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    public String readN(int count) throws IOException {
	char[] buf = new char[count];

	read(buf, 0, count);

	return new String(buf);
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    public boolean ready() throws IOException {
	try {
	    return reader.ready();
	} catch (IOException e) {
	    throw new ParseException(e, this);
	} 
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    public String readIdentifier() throws IOException {
	StringBuffer result = new StringBuffer();

	while ((next >= '0' && next <= '9' && result.length() > 0) 
	       || (next >= 'a' && next <= 'z') 
	       || (next >= 'A' && next <= 'Z') || (next >= 128)) {
	    result.append((char) read());
	}

	return result.toString();
    } 

    /**
     * reads a String until any of the given stopChars is reached.
     * The stopChar itself is not consumed
     */
    public String readTo(String stopChars) throws IOException {
	StringBuffer result = new StringBuffer();

	while (next != -1 && stopChars.indexOf((char) next) == -1) {
	    result.append((char) read());
	}

	return result.toString();
    } 

    /**
     * Method declaration
     *
     *
     * @param stopChar
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    public String readTo(char stopChar) throws IOException {
	StringBuffer result = new StringBuffer();

	while (next != -1 && next != stopChar) {
	    result.append((char) read());
	}

	return result.toString();
    } 

    /**
     * Method declaration
     *
     *
     * @param stopChars
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    public int skipTo(String stopChars) throws IOException {
	while (next != -1 && stopChars.indexOf((char) next) == -1) {
	    read();
	}

	return next;
    } 

    /**
     * Method declaration
     *
     *
     * @param acceptChars
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    public String readWhile(String acceptChars) throws IOException {
	StringBuffer result = new StringBuffer();

	while (next != -1 && acceptChars.indexOf((char) next) != -1) {
	    result.append((char) read());
	}

	return result.toString();
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @see
     */
    public int peek() {
	return next;
    } 

    /**
     * Method declaration
     *
     *
     * @return
     *
     * @throws IOException
     *
     * @see
     */
    public int read() throws IOException {
	int result = next;

	if (next != -1) {
	    try {
		next = reader.read();
	    } catch (IOException e) {
		throw new ParseException(e, this);
	    } 
	} 

	if (next == 10) {
	    lineNumber++;
	    columnNumber = 0;
	} else {
	    columnNumber++;
	}

	return result;
    } 

}

